Scriptname RF:FuelHandlerQuest Extends Quest

Group Quests
    SQ_PlayerShipScript Property SQ_PlayerShip Auto
    DialogueShipServicesScript Property DialogueShipServices Auto Const mandatory
EndGroup

Group AVs
    ActorValue Property SpaceshipGravJumpFuel auto const mandatory
    ActorValue Property SpaceshipPartsMass Mandatory Const Auto
    ActorValue Property SpaceshipGravJumpCalculation Mandatory Const Auto
EndGroup

Group System
    Perk Property _RF_Perk Mandatory Const Auto
    { This will hold our SPEL ability to control grav jump range }
    GlobalVariable Property _RF_Sys_ModEnabled Mandatory Const Auto
    GlobalVariable Property _RF_Sys_AllowTravel Mandatory Const Auto
    GlobalVariable Property _RF_Sys_Hardcore Mandatory Const Auto
    GlobalVariable Property _RF_Sys_Verbose Mandatory Const Auto
    ;DEBUG
    Potion Property DebugPotion1 Mandatory Const Auto
    Potion Property DebugPotion2 Mandatory Const Auto
    Potion Property DebugPotion3 Mandatory Const Auto
    Potion Property DebugPotion4 Mandatory Const Auto
    GlobalVariable Property _RF_DTV Mandatory Const Auto
    { This lets us force certain random events }
    ;States
    GlobalVariable Property _RF_State_CanActivateCargoPanel Mandatory Const Auto
    { This smartly determines which action to take }
EndGroup

Group Travel
    GlobalVariable Property _RF_DistanceRestriction Mandatory Const Auto
    { Severity tied to SPEL record. 0:None 1:40% 2:60% 3:90% (should be functionally inter-system only) }
    GlobalVariable Property _RF_GravDebuff Mandatory Const Auto
    { 1: 2x 2: 4x 3:8x }
    Location[] Property StarSystemsCore Mandatory Const Auto
    { 0:SOL 1:AC 2:CHEYENNE 3:VOLII }
    Location[] Property StarSystemsPeriphery Mandatory Const Auto
    { Fuel is drained here according to whether it's to/from here }
    Location[] Property StarSystemsOuterRim Mandatory Const Auto
    { Fuel is drained moderately when traveling to/from here }
    ;Location[] Property StarSystemsWastes Mandatory Const Auto
    ;{ This is just the rest of the universe }
    Keyword Property LocTypeMajorOrbital Mandatory Const Auto
    Keyword Property LocTypeOutpost Mandatory Const Auto
    Keyword Property LocTypeStarSystem Mandatory Const Auto
    Keyword Property LocTypeStarstationExterior Mandatory Const Auto
    Location[] Property UCStarstations Mandatory Const Auto
    { This can trip us up if we are in combat near one, but... I don't think that's likely. }
    Faction Property CrimeFactionCrimsonFleet Mandatory Const Auto
	Faction Property CrimeFactionFreestar Mandatory Const Auto
	Faction Property CrimeFactionParadiso Mandatory Const Auto
	Faction Property CrimeFactionRedMile Mandatory Const Auto
	Faction Property CrimeFactionUC Mandatory Const Auto
    ConditionForm Property _RF_InCooldownState Mandatory Const Auto
    { Indicates we are cooling down from a jump }
    Spell Property _RF_GravDriveDetectiveSpell Mandatory Const Auto
    Spell Property _RF_GravDriveOverheatSpell Mandatory Const Auto
    Formlist Property _RF_TwinStarFormlist Mandatory Const Auto
EndGroup

Group Ship
    Keyword Property LocTypeSettlement Mandatory Const Auto
    Keyword Property ShipManufacturerDeimos Mandatory Const Auto
    Keyword Property ShipManufacturerHopeTech Mandatory Const Auto
    Keyword Property ShipManufacturerNova Mandatory Const Auto
    Keyword Property ShipManufacturerStroud Mandatory Const Auto
    Keyword Property ShipManufacturerTaiyo Mandatory Const Auto
    Keyword[] Property ShipModuleClasses Mandatory Const Auto
    { 0-3 A-M }
    GlobalVariable[] Property ShipModuleClassMults Mandatory Const Auto
    { 0:40 1:70 2:120 3:220}
    ConditionForm[] Property AstrodynamicsRank Mandatory Const Auto
    { 0:none etc }
    GlobalVariable[] Property AstrodynamicsMults Mandatory Const Auto
    { These can match vanilla for now }
    LocationRefType Property Ship_PilotSeat_RefType Mandatory Const Auto
    { To check if we sat in pilot's seat }
    ConditionForm Property _RF_COND_PlayerShipInterior Mandatory Const Auto
    { I could use ENV_CND_InSealedShip but I want to be sure it's our ship... for now I guess }
    GlobalVariable Property _RF_Val_SiphoningCrimeGoldValue Mandatory Const Auto
    Potion Property _RF_FuelSiphon Mandatory Const Auto
    { I would like to restrict functionality to this item somehow. Maybe greater efficiency? }
    Potion[] Property FuelTanks Mandatory Const Auto
    { In ascending order of quality }
    MiscObject property InorgCommonHelium3 auto const mandatory
    { Only for emergencies! }
    Quest Property MQ101 Mandatory Const Auto
    WwiseEvent Property SoundWhenTriggeringSmartFuel Mandatory Const Auto
EndGroup

Group Messages
    Message Property _RF_Message_FuelRemainingOK Mandatory Const Auto
    { Above 50% at breakpoints }
    Message Property _RF_Message_FuelRemainingLOW Mandatory Const Auto
    { Below 50% above 25% }
    Message Property _RF_Message_FuelRemainingNG Mandatory Const Auto
    { Below 25% }
    Message Property _RF_Alert_GravJumpDisabled Mandatory Const Auto
    Message Property _RF_Alert_GravJumpEnabled Mandatory Const Auto
    Message Property _RF_Alert_Map_ShipDamage Mandatory Const Auto
    Message Property _RF_Alert_ShipDamage Mandatory Const Auto
    Message Property _RF_SiphonFuelWarning Mandatory Const Auto
    Message Property _RF_Alert_TooManyJumps Mandatory Const Auto
EndGroup

Group Tutorial
    Message Property _RF_tutorial_outoffuel Mandatory Const Auto
EndGroup

Actor PlayerRef
SpaceshipReference MyShip
SpaceshipReference SiphonedShip

int StartupTimer = 1

Location[] TwinStars

float ShipFuelAvailable = 1.0 ; Actual AV amount. Updated in GetFuelLevel     
float ShipMaximumRange = 1.0 ; This is in light years. I probably don't need it
int LastPercent = 100 ; Updated in Restrictions()
bool JumpWasIntrasystem = false ; Updated in GetTravelDistance. True = No Grav Stuff
bool WasGravJumpDisabled = false ; Updated in Restrictions() to avoid message spam
bool SpoolingUp = false ; Set in GravJumpHandler() in state 0
int SequentialFastJumps = 0 ; Set in whatever handler it is
bool TwinJump = False ; Drain will be random under 20 if this is true
bool MyOldShipSteal = False ; In exactly one scenario a different message will show
bool ShipInWastes = false ; Gets set in CheckRestrictions() to trigger our fallback travel limiters

float FuelToDrawSender = 0.0 ; This would be called from the DLL sender script and fed in 
float SiphonableFuel = 1.0 ; This is updated OnDocking

;Used for exceptions or out-of-fuel scenarios
bool UsingEmergencyJump = false ; Configurable... somewhere
float LastFuelAmount = 1.0 ; Filled if we are adding emergency jump functionality
int EmergencyJumps = 0 ; Incremented each time we perform an emergency jump
bool showtutorials = true ; set to false after first nag message shows

int TutorialCount = 0 ; called in Notify events to periodically bug the player

bool AutoRefuelRunning = false ; Prevent spam on panel activate

;Counts for breakpoints
int BreakHigh = 70
int BreakNorm = 40
int BreakMerg = 25
int BreakCrit = 10

InputEnableLayer FastTravelInputLayer ; May skip this one entirely
InputEnableLayer FarTravelInputLayer
InputEnableLayer GravJumpInputLayer

;System
bool RF = true ; Is the mod running
bool DEBUGON = true ; Debug handler
bool DEBUGNOTIFY = false ; I can't tell how fast the scripts are running with this spamming me lol
bool REALFUELON = false ; I don't want to use this until the scaling is set in stone
bool DEBUGSTART = false ; The only debug implement is the free shit on startup


;----------------------------------------------------
;----------------- DEBUG AND SYSTEM ----------------- 
;----------------------------------------------------


;Generic debug function that can be disabled for prod
Function DBG(String asTextToPrint = "Debug Error!")
    If DEBUGON
        Debug.Trace("RF FuelHandler: " + asTextToPrint)
        IF DEBUGNOTIFY ; Can log in realtime to avoid alt-tabbing constantly
            Debug.Notification("RFUEL: " + asTextToPrint)
        EndIF
    EndIF
EndFunction

Function DBGVB(String asTextToPrint = "Debug Error!")
    If DEBUGON
        Debug.Notification("RFUEL: " + asTextToPrint)
    EndIF
EndFunction

;I may have to call this in locchange
bool Function HandleEmergencyJump()
    bool DidDamageOccur = false ; If this happens we skip the fuel withdrawal phose
    If UsingEmergencyJump ; Indicates we were on critical fuel before jumping
        ActorValue Hull = Game.GetHealthAV()
        DidDamageOccur = CoinFlip()
        If DidDamageOccur
            float Max = MyShip.GetBaseValue(Hull) * 0.35
            float Damage = Max * GetVariance(0.2)
            HelpMessage(_RF_Alert_ShipDamage)
            Utility.Wait(0.1)
            MyShip.DamageValue(Hull, Damage) ; This might kill you - cool!
            MyShip.SetPartPower(3, -1, 0) ; test if this is reversible
            ;DisableAndNotify(false) ; This just gets reset - d'oh
            ;_RF_GravDriveOverheatSpell.Cast(PlayerRef)
        EndIf
    EndIf
    Return DidDamageOccur
EndFunction

bool Function HandleOverheatEnd()
    bool OK = false
    ActorValue Hull = Game.GetHealthAV()
    float HP = MyShip.GetValuePercentage(Hull)
    If HP > 0.5
        OK = True
    EndIF
    Return OK
EndFunction

;true indicates verbose
bool Function VB()
    bool verbose = false
    if _RF_Sys_Verbose.GetValue() == 1
        verbose = true
    endif
    return verbose
EndFunction

;true indicates easy mode
bool Function TRV()
    bool easytrails = false
    if _RF_Sys_AllowTravel.GetValue() == 1
        easytrails = true
    endif
    return easytrails
EndFunction

; Greater weight value increases chance of failing flip
bool Function CoinFlip(int aiWeight = 50)
    bool Heads = true
    int Coin = Utility.RandomInt(0, 100)
    If Coin <= aiWeight
        Heads = false
    EndIf
    Return Heads
EndFunction

; Gives us a +- mult to use for randomization
float Function GetVariance(float afMax = 0.25)
    bool Random = Coinflip()
    float Divergence = 1.0
    float RandomAmount = Utility.RandomFloat(0.01, afMax)
    If Random
        Divergence = 1.0 - RandomAmount
    Else
        Divergence = 1.0 + RandomAmount
    EndIF
    return Divergence
EndFunction

Function HelpMessage(Message asMessageToShow, int Duration = 7)
    Utility.Wait(2)
    If asMessageToShow
        asMessageToShow.ShowAsHelpMessage(asEvent = 1, afDuration = Duration, afInterval = 1, aiMaxTimes = 1)
    EndIF
EndFunction

; True will prevent us from grav jumping... False lets us grav jump.
Function DisableGravJump(bool abDisabling)
    If abDisabling
        If (GravJumpInputLayer == None)
            GravJumpInputLayer = InputEnableLayer.Create()
            GravJumpInputLayer.EnableGravJump(False)
            DBG("Disabling Grav Jump")
        Else
            DBG("EXCEPTION - DisableGravJump called with no active input layer present!")
        EndIF
    Else
        If (GravJumpInputLayer != None)
            GravJumpInputLayer.EnableGravJump(True)
            GravJumpInputLayer.Delete()
            GravJumpInputLayer = None
            DBG("Enabling Grav Jump")
        EndIf
    EndIF
EndFunction

; This will disable inter-system flight as well. Probably the best use case for severe stranding. 
Function DisableFarTravel(bool abDisabling)
    If abDisabling
        FarTravelInputLayer = InputEnableLayer.Create()
        FarTravelInputLayer.EnableFarTravel(False)
        DBG("Disabling Far Travel")
    Else
        If (FarTravelInputLayer != None)
            FarTravelInputLayer.EnableFarTravel(True)
            FarTravelInputLayer.Delete()
            FarTravelInputLayer = None
            DBG("Enabling Far Travel")
        EndIf
    EndIF
EndFunction

; Testing: This will disable everything, apparently. Too broad.
Function DisableFastTravel(bool abDisabling)
    If abDisabling
        FastTravelInputLayer = InputEnableLayer.Create()
        FastTravelInputLayer.EnableFastTravel(False)
        DBG("Disabling Fast Travel")
    Else
        If (FastTravelInputLayer != None)
            FastTravelInputLayer.EnableFastTravel(True)
            FastTravelInputLayer.Delete()
            FastTravelInputLayer = None
            DBG("Enabling Fast Travel")
        EndIf
    EndIF
EndFunction


;-------------------------------------------------
;----------------- TRAVEL SYSTEM -----------------
;-------------------------------------------------

; Thanks Google!
bool Function is_valid_pair_difference(int x, int y)
    bool pair = false
    int diff = Math.Abs(x - y) as int
    int minVal = Math.Min(x, y) as int
    if diff == 1 && (minVal % 2) == 0
        pair = true
    endif
    DBG("pairdiff got " + pair + " from x " + X + " and y " + y)
    return pair
EndFunction

bool Function CheckForTwinStars(Location aOldSys, Location aNewSys)
    bool Pair = False
    int NewSysLoc = TwinStars.Find(aNewSys)
    int OldSysLoc = TwinStars.Find(aOldSys)
    If NewSysLoc + OldSysLoc >= 0 ; This is only true if *both* are twin stars
        Pair = is_valid_pair_difference(NewSysLoc, OldSysLoc)
    EndIF
    DBG("CFTS(TM) got " + Pair + " from new " + aNewSys + ": " + NewSysLoc + " and old " + aOldSys + ":" + OldSysLoc)
    Return Pair
EndFunction

;This returns what kind of system we are in
;0:Core 1:Periphery 2:Outer etc
int Function GetSysType(Location akLocation)
    int Type = 3
    If StarSystemsCore.Find(akLocation) >= 0
        Type = 0
    ElseIf StarSystemsPeriphery.Find(akLocation) >= 0
        Type = 1
    ElseIf StarSystemsOuterRim.Find(akLocation) >= 0
        Type = 2
    Else
        Type = 3
    EndIF
    ;DBG("GetSysType returned " + Type + "from Location " + akLocation)
    Return Type
EndFunction

float Function CompareTypes(int aiO, int aiN)
    int Pen = 0
    int RD = Math.abs(aiN - aiO) as int
    If aiO > aiN    ; towards center - lower



    Else            ; towards periphery - higher



    EndIF
EndFunction

int Function CheckLocationDistance(Location aDepartureLocation, Location aArrivalLocation)
    ShipInWastes = False
    JumpWasIntrasystem = False ; Almost forgot this...
    int DistanceMultFlat = 1
    Location[] NewLocArray = aArrivalLocation.GetParentLocations(LocTypeStarSystem)
    Location[] OldLocArray = aDepartureLocation.GetParentLocations(LocTypeStarSystem)
    Location MyNewLoc = NewLocArray[0]
    Location MyOldLoc = OldLocArray[0]
    If MyNewLoc != MyOldLoc
        TwinJump = CheckForTwinStars(MyOldLoc, MyNewLoc)
        If !TwinJump
            int TypeNew = GetSysType(MyNewLoc)
            int TypeOld = GetSysType(MyOldLoc)
            float Dist = CompareTypes(TypeOld, TypeNew)
            DistanceMultFlat = TypeNew + TypeOld ; This is the original logic - we will feed in Dist as a float in future
            TwinJump = False
        EndIF
    Else
        JumpWasIntrasystem = true
    EndIf
    DBG("CheckLocationDistance got " + DistanceMultFlat + " from OldLoc " + MyOldLoc + " and NewLoc " + MyNewLoc)
    Return DistanceMultFlat
EndFunction

;This is the entire core of our travel system disabler.
;It currently operates purely on fuel percent but could be extended for skills.
Function CheckRestrictions()
    UsingEmergencyJump = False
    Bool EZ = TRV()
    Bool Disable = False
    Bool Notify = false
    int Percent = GetFuelLevelInt()
    int Restriction = 0
    int GravDebuff = 0
    float Mult = 1.0
    If Percent > ( BreakHigh * Mult) ; Ease any restrictions present
        Restriction = 0
    ElseIf Percent > ( BreakNorm * Mult ) ; 80% range
        Restriction = 1
    ElseIf Percent > ( BreakMerg * Mult ) ; 70% range
        Restriction = 2
    Else ; Disable inter-system travel entirely or 60% range in not hardcore mode
        Restriction = 3
        If ( Percent > BreakCrit ) && !EZ
            UsingEmergencyJump = True
        Else
            Disable = true
            if showtutorials
                _RF_tutorial_outoffuel.Show()
                showtutorials = false
            endif
        EndIf
    EndIf
    If LastPercent - Percent > 40 || Percent < BreakNorm || VB()
        SmartNotify(Percent)
    EndIF
    LastPercent = Percent ; Gotta set this after...
    If !EZ
        If Disable
            DisableAndNotify() ; This just keeps the system alive and ticks up the tutorial counter
        Else
            EnableAndNotify()
        EndIf
    Else
        DBG("CheckRestrictions killing handler with TRV True")
        Restriction = 0
        UsingEmergencyJump = False
        EnableAndNotify()
    EndIF
     _RF_DistanceRestriction.SetValue(Restriction) ; This alters our grav drive max range to: 100%, 50%, 35%, and of course at 3 it is disabled entirely.
    _RF_GravDebuff.SetValue(Restriction) ; This hampers our Grav Drive spool up speed. Conveniently, it uses the same values
    DBG("CheckRestrictions: Tier" + Restriction + " from " + Percent + "% fuel.") ; " with spool speed " + MyShip.GetValue(SpaceshipGravJumpCalculation))
EndFunction

Function EnableAndNotify()
    If (GravJumpInputLayer != None)
        DisableGravJump(false)
        WasGravJumpDisabled = False
        HelpMessage(_RF_Alert_GravJumpEnabled)
    EndIF
EndFunction

Function DisableAndNotify(bool abNotify = true)
    bool Notify = false
    If (GravJumpInputLayer == None)
        DisableGravJump(true)
        WasGravJumpDisabled = True
        Notify = True
    Else
        DBG("DisableAndNotify: incrementing spam counter")
        TutorialCount += 1
    EndIf
    If abNotify
        IF Notify || TutorialCount > 3
            HelpMessage(_RF_Alert_GravJumpDisabled) ; This was wrong...
            TutorialCount = 0
        EndIf
    EndIf
EndFunction

Function SmartNotify(int aiPercent = 120)
    Utility.Wait(2.8)
    If aiPercent > 100 ; indicates failure or default
        aiPercent = GetFuelLevelInt()
    EndIf
    If aiPercent >= BreakNorm
        _RF_Message_FuelRemainingOK.Show(aiPercent)
    ElseIf aiPercent >= BreakCrit
        _RF_Message_FuelRemainingLOW.Show(aiPercent)
    Else
        _RF_Message_FuelRemainingNG.Show(aiPercent)
        Utility.Wait(6)
        _RF_Message_FuelRemainingNG.Show(aiPercent)
    EndIF
EndFunction

; Pretty simple actually
; Not called yet. gotta sort emergency handling first!!!!
Function CooldownHandler()
    If !TRV()
        If _RF_InCooldownState.IsTrue(PlayerRef)
            SequentialFastJumps += 1
            DBG("CooldownHandler called with InCooldownState True" + SequentialFastJumps)
        Else
            SequentialFastJumps = 0
            DBG("CooldownHandler called with InCooldownState False. Clearing FastJumps " + SequentialFastJumps)
        EndIf
        If SequentialFastJumps > 4 || _RF_DTV.GetValue() == 12
            IF CoinFlip() || _RF_DTV.GetValue() == 12
                DBG("Firing overheat spell...")
                _RF_GravDriveOverheatSpell.Cast(PlayerRef)
                HelpMessage(_RF_Alert_TooManyJumps)
            EndIF
        EndIF
    EndIf
EndFunction


;--------------------------------------------------
;----------------- FUELING SYSTEM -----------------
;--------------------------------------------------


;This is our scripted internal handler to add fuel. Add all of it or add some, that's it. No notifs
int Function Realfuel(bool abComplete, float afAmount = 1.0)
    DBG("RealFuel called on " + MyShip)
    Float FuelMissing = MyShip.GetBaseValue(SpaceshipGravJumpFuel) - MyShip.GetValue(SpaceshipGravJumpFuel)
    int Withdrawn = 0
    ;DBG("Realfuel thinks we need " + FuelMissing + " units ")
    If abComplete
        MyShip.RestoreValue(SpaceshipGravJumpFuel, FuelMissing)
        DBG("RealFueled ship COMPLETE " + FuelMissing + " RESULTING " + CheckFuelAmount(false) )
    Else
        float AmountToFill = Math.Clamp(afAmount, 0, FuelMissing)
        MyShip.RestoreValue(SpaceshipGravJumpFuel, AmountToFill)
        Withdrawn = AmountToFill as int
        DBG("RealFueled ship CLAMP " + AmountToFill + " FROM " + afAmount + " RESULTING " + CheckFuelAmount(false) + " with Withdrawn " + Withdrawn)
    EndIf
    GetFuelLevel()
    CheckAltActions()
    CheckRestrictions() ; I think this is OK to have here.
    DialogueShipServices.UpdateFuelGlobals()
    Return Withdrawn
EndFunction

;This is done silently, and notifications are all handled in CheckRestrictions
Function HandleDrawingFuel(int aiDistance)

    If FuelToDrawSender != 0
        MyShip.DamageValue(SpaceshipGravJumpFuel, FuelToDrawSender)
    Else
        ; I want to draw between 40 and 300 fuel for most voyages.
        ; A-B-C-M corresponds to 35 > 70 > 120 > 220 for starters.
        Float FuelToDrain = 1.0
        float Variance = GetVariance()
        float ADSkill = CheckAstrodynamics()
        float InShip = MyShip.GetValue(SpaceshipGravJumpFuel)
        int ReactorClass = ShipModuleClasses.Find(MyShip.GetReactorClassKeyword())
        int BaseDraw = (ShipModuleClassMults[ReactorClass] as GlobalVariable).GetValue() as int
        int ExtraDraw = 12 ; This is a smidge high for short jumps still...
        float DistanceDerived = ( aiDistance / 2 ) * 0.75 ; This should range from 1.4 to 6
        float ShipMult = GetShipMassMult()
        If TwinJump
            FuelToDrain = BaseDraw * Utility.RandomFloat(0.25, 0.4)
            DBG("HandleDrawingFuel cutting to " + FuelToDrain + " with TwinJump " + TwinJump)
        Else
            FuelToDrain = (BaseDraw + (ExtraDraw * ShipMult * DistanceDerived)) * Variance * ADSkill
        EndIF
        float FuelClamped = Math.Clamp( FuelToDrain , 0 , InShip )
        DBG("HandleDrawingFuel got drain REAL:" + FuelToDrain + " amount of Clamped " + FuelClamped as int + " units from Base:" + BaseDraw + " + (Extra:" + ExtraDraw + " x Size:" + ShipMult + " x Distance:" + DistanceDerived + " ) x Variance:" + Variance + " x Skill:" + ADSkill)
        MyShip.DamageValue(SpaceshipGravJumpFuel, FuelClamped)
        CheckAltActions()
        TwinJump = False    
    EndIf

    DialogueShipServices.UpdateFuelGlobals()

EndFunction

; Handler function to remove all of our fuel. Only used on transferring vessels!
Function ForceDrainFuel(bool abComplete, Float afAmount = 0.0)
    MyShip = SQ_PlayerShip.PlayerShip.GetShipRef()
    float FuelInTank = CheckFuelAmount()
    float FuelToDrain = 1.0
    If abComplete
        FuelToDrain = FuelInTank
    Else
        if afAmount > 0
            FuelToDrain = Math.Clamp(afAmount, 0, FuelInTank)
        EndIf
    EndIf
    MyShip.DamageValue(SpaceshipGravJumpFuel, ( FuelToDrain - 5 ) )
    DBG("ForceDrained " + FuelToDrain + "from " + FuelInTank) ; No updatefuelglobals here as we only call this when we are gonna fill it again
EndFunction

;Called from our smart activate on the panel
bool Function RefuelFromHold()
    Utility.Wait(0.4)
    Debug.Notification("Searching hold for valid refuel materials...")
    int i = 0
    bool Found = false
    bool Success = false
    Form Tank
    While i < FuelTanks.Length
        IF !Found
            If CheckHold(FuelTanks[i])
                Tank = FuelTanks[i] as Form
                DBG("RefuelFromHold Loop check found a " + Tank)
                Found =  True
            EndIf
        EndIF
        i += 1
    EndWhile
    If Tank
        MyShip.RemoveItem(Tank, 1, true)
        PlayerRef.AddItem(Tank, 1, true)
        Debug.Notification("Refuel tank found. Beginning load procedure.")
        Utility.Wait(5)
        PlayerRef.EquipItem(Tank, false, true)
        Success = True
    Else
        DBG("RefuelFromHold Progressing to RefuelFromHoldManual")
    EndIF
    Return Success
EndFunction

bool Function RefuelFromHoldManual()
    float FuelWeNeed = CheckFuelAmount(true)
    int HeliumInHold = MyShip.GetItemCount(InorgCommonHelium3)
    DBG("RFHM thinks we need " + FuelWeNeed + " with available he3 " + HeliumInHold)
    float Efficiency = 0.75 * GetVariance(0.1)
    float HeliumToConsume = Math.Clamp(HeliumInHold, 0, FuelWeNeed)
    float HeliumWeGet = HeliumToConsume * Efficiency
    DBG("RFHM got HeliumWeGet " + HeliumWeGet + " with FuelWeNeed " + FuelWeNeed + " and efficiency" + Efficiency + " using he3 " + HeliumToConsume )
    If HeliumWeGet > 15 ; This is just a nitpick but if it keeps causing problems I will cut it
        Realfuel(false, HeliumWeGet)
        MyShip.RemoveItem(InorgCommonHelium3, HeliumToConsume as int, true)
        Utility.Wait(4)
        Debug.Notification( "Using " + HeliumToConsume as int + " He-3 from ship's hold.")
        Utility.Wait(6)
        SmartNotify(GetFuelLevelInt())
        return true
    Else
        ;Debug.Notification("Insufficient He-3 found in hold.")
        DBG("RefuelFromHoldManual failed. No fueling conducted with he3 in hold " + HeliumInHold)
        return false
    EndIF
EndFunction


;-------------------------------------------------
;----------------- SHIP CHECKERS -----------------
;-------------------------------------------------


; Quick handler to print our current fuel level
; This should be called before any math as it fills our ShipFuelAvailable value!!!
float Function GetFuelLevel()
    MyShip = SQ_PlayerShip.PlayerShip.GetShipRef() ; This causes ship change stuff to fail?
    float FuelAvailable = 1.0
    float FuelNow = MyShip.GetValue(SpaceshipGravJumpFuel) + 0.1 
    float FuelAll = MyShip.GetBaseValue(SpaceshipGravJumpFuel)
    FuelAvailable = FuelNow / FuelAll
    ShipFuelAvailable = FuelNow ; This is called after every fuel drain event so it's going to be accurate
    DBG( "PlayerShip Float: " + FuelAvailable + ". Stored value " + ShipFuelAvailable as int)
    return FuelAvailable
EndFunction

int Function GetFuelLevelInt()
    float FuelAvailable = 1.0
    float FuelNow = MyShip.GetValue(SpaceshipGravJumpFuel)
    float FuelAll = MyShip.GetBaseValue(SpaceshipGravJumpFuel)
    FuelAvailable = ( MyShip.GetValue(SpaceshipGravJumpFuel) / ( MyShip.GetBaseValue(SpaceshipGravJumpFuel) + 0.1 ) * 100 ) as int
    DBG( "PlayerShip Percent: " + FuelAvailable as int + "% " + FuelNow as int + "/" + FuelAll as int)
    Return FuelAvailable as int
EndFunction

; This should be called before *anything* as there is no real automatic update
SpaceshipReference Function GetShip()
    MyShip = SQ_PlayerShip.PlayerShip.GetShipRef()
    DBG( "RF found Playership as " + MyShip )
    Return MyShip
EndFunction

;New math based around the ExtraDraw method
float Function GetShipMassMult()
    float MassScale = 1.0
    float Mass = MyShip.GetValue(SpaceshipPartsMass)
    MassScale = ( Mass / 350 + 0.5) ; This should range from 0.5 to ~5.2
    ;DBG("GetShipMassMult got " + MassScale + " from ship size " + Mass)
    Return MassScale
EndFunction

;This returns a float value to match the vanilla skill % reduction
Float Function CheckAstrodynamics()
    float AmountToLower = 1.0
    int RankFound = 0
    int i = 0
    While i < AstrodynamicsRank.length
        If AstrodynamicsRank[i].IsTrue(PlayerRef)
            AmountToLower = AstrodynamicsMults[i].GetValue()
            RankFound = i
        EndIf
        i += 1
    EndWhile
    ;DBG("CheckAstrodynamics returning " + AmountToLower + " from rank:" + RankFound)
    Return AmountToLower
EndFunction

;Mostly for curiosity
float Function GetGravRange()
    float Range = 1.0
    Range = MyShip.GetGravJumpRange()
    If (GravJumpInputLayer != None)
        DBG("GetGravRange thinks Grav Jump is disabled")
    EndIf
    ShipMaximumRange = Range 
    Return Range
EndFunction

Function SiphonClear()
    SiphonableFuel = 0
    SiphonedShip = None
    MyOldShipSteal = False
EndFunction

Function TriggerBounty(Faction akFaction)
    float SiphonPenalty = _RF_Val_SiphoningCrimeGoldValue.GetValue()
    If akFaction
        akFaction.ModCrimeGold(SiphonPenalty as int, false)
        DBG("Triggered bounty of " + SiphonPenalty + " with " + akFaction)
    Else
        Faction Reporting = CheckShipCrime()
        If Reporting
            Reporting.ModCrimeGold(SiphonPenalty as int, false)
            DBG("Triggered bounty of " + SiphonPenalty + " with " + Reporting)
        Else
            DBG("TriggerBounty did not find faction")
        EndIF
    EndIf
EndFunction


;-------------------------------------------------
;----------------- SYSTEM/EVENTS -----------------
;-------------------------------------------------


Function ImportTwinStars()
    Int listSize = _RF_TwinStarFormlist.GetSize()
    TwinStars = new Location[listSize]
    Int i = 0
    While i < listSize
        TwinStars[i] = _RF_TwinStarFormlist.GetAt(i) as Location
        i += 1
    EndWhile

    DBG("Imported twinsformlist")

    ;CheckTwins()
EndFunction

Function CheckTwins()
    int i = 0
    While i < TwinStars.Length
        DBG("twincheck got " + TwinStars[i] + "at " + i)
        i += 1
    EndWhile
EndFunction

;This should get our current fuel level and set up our ship
Function HandleModStartup()
    RF = True
    int Fuel = CheckFuelAmount(false) as int
    GetFuelLevel() ; This should be here to prevent exceptions on startup, obviously
    ImportTwinStars()
    Debug.Notification("Real Fuel running. " + Fuel + " Fuel available.")
EndFunction

;Quick refresh
Function PerkRefresh(bool abEnable)
    IF abEnable
        If !PlayerRef.HasPerk(_RF_Perk)
            PlayerRef.AddPerk(_RF_Perk)
        EndIf
    Else
        If PlayerRef.HasPerk(_RF_Perk)
            PlayerRef.RemovePerk(_RF_Perk)
        EndIf
    EndIF
EndFunction

;To check if we need to be able to take cargo panel actions
Function CheckAltActions()
    int activate = 0
    If GetFuelLevelInt() < BreakHigh
        activate = 1   
    EndIf
    _RF_State_CanActivateCargoPanel.SetValue(activate)
    ;DBG("Alt activate state: " + activate)
EndFunction

Event OnQuestInit()
    StartTimer(3, StartupTimer)
EndEvent

Event OnTimer(int aiTimerID)
    PlayerRef = Game.GetPlayer()
    GetShip()
    RegisterForRemoteEvent(PlayerRef, "OnPlayerLoadGame")
    RegisterForRemoteEvent(PlayerRef, "OnExitShipInterior")
    RegisterForRemoteEvent(PlayerRef, "OnHomeShipSet")
    RegisterForRemoteEvent(PlayerRef, "OnPlayerModifiedShip")
    RegisterForRemoteEvent(PlayerRef, "OnPlayerBuyShip")
    RegisterForRemoteEvent(PlayerREf, "OnSit")
    ;RegisterForCustomEvent(SQ_PlayerShip, "SQ_PlayerShipChanged") ; Seems to not be available, whatever
    RegisterForMenuOpenCloseEvent("GalaxyStarMapMenu")
    PlayerRef.AddPerk(_RF_Perk)
    If DEBUGSTART
        PlayerRef.AddItem(DebugPotion1, 3, true)
        PlayerRef.AddItem(DebugPotion2, 3, true)
        PlayerRef.AddItem(DebugPotion3, 3, true)
        PlayerRef.AddItem(DebugPotion4, 3, true)
        Debug.Notification("Fueling items added for testing.")
    EndIf
    HandleModStartup()
EndEvent

;Mostly just for notifications at this point, I don't need it for logic
Event OnMenuOpenCloseEvent(String asMenuName, Bool abOpening)
    If asMenuName == "GalaxyStarMapMenu" && RF
        If abOpening
            int Percent = GetFuelLevelInt()
            If Percent >= BreakCrit
                _RF_Message_FuelRemainingOK.Show(Percent)
            Else
                ;Debug.Notification("Grav Jumping in this state may severely damage your ship.")
                HelpMessage(_RF_Alert_Map_ShipDamage)
            EndIf
        Elseif !abOpening
            If !SpoolingUp
                FuelToDrawSender = 0 ; This may or may not clear the thing entirely when we go to jump. Check spoolup?
            Else
                DBG("StarMapMenu FuelToDraw clearance canceled with SpoolingUp:" + SpoolingUp)
            EndIF
        EndIF
    EndIf
EndEvent

;Transfers our old fuel amount to the new ship exactly (overflow is lost)
Event Actor.OnPlayerModifiedShip(Actor akSender, SpaceshipReference akShip)
    DBG("PlayerModifiedShip on firing has current fuel " + CheckFuelAmount() + " vs last stored " + ShipFuelAvailable )
    If RF
        float NewFuel = CheckFuelAmount()
        GetShip()
        ForceDrainFuel(true)
        Realfuel(false, ShipFuelAvailable)
        int Percent = GetFuelLevelInt()
        SmartNotify(Percent) ; I love this function
        DBG("PlayerModifiedShip after mod has current fuel " + CheckFuelAmount() + " vs last stored " + ShipFuelAvailable )
    EndIf
EndEvent

;This is technically different enough from the above that I don't want to merge code
Function HandleShipPurchase(SpaceshipReference akShip)
    If RF
        Utility.Wait(4) ; God only knows how long the vanilla jank takes to fire
        DBG("HandleShipPurchase fired with current fuel " + CheckFuelAmount() + " vs last stored " + ShipFuelAvailable )
        Utility.Wait(2)
        GetShip() ; This refreshes our ship, we shouldn't need anything from the old one at this point
        float Gratis = MyShip.GetValue(SpaceshipGravJumpFuel) * 0.75 * GetVariance()
        ForceDrainFuel(true)
        Realfuel(false, Gratis)
        int FuelInNew = CheckFuelAmount() as int
        Debug.Notification("Purchased ship came with " + FuelInNew + " complementary He-3.")
        DBG("HandleShipPurchase after mod has current fuel " + CheckFuelAmount() + " vs last stored " + ShipFuelAvailable )
    EndIf
EndFunction

;Fires the above
Event Actor.OnPlayerBuyShip(Actor akSender, SpaceshipReference akShip)
    HandleShipPurchase(akShip)
EndEvent

Event Actor.OnHomeShipSet(Actor akSender, SpaceshipReference akShip, SpaceshipReference akPrevious)
    DBG("Ship changed with previous " + akPrevious + " and current " + akShip)
    Location PlayerIsIn = PlayerRef.GetCurrentLocation()
    If PlayerIsIn.HasKeyword(LocTypeOutpost) || PlayerIsIn.HasKeyword(LocTypeSettlement)
        ; This actually only happens in settlements, so we use OnSit()
        DBG("Player probably bought a ship - confirm.")
    EndIf
EndEvent


;Sanity checks our ship
Event Actor.OnPlayerLoadGame(Actor akSender)
    If RF
        GetShip()
        GetFuelLevel()
    EndIF
EndEvent

;This handles just about everything and should be fast and robust (hence arrays in CheckDistance etc)
Function HandleLocChange(Location akOldLoc, Location akNewLoc)
    If RF
        GetShip()
        ;Padding is a hack for now - this function should set InterSystem bool and return a useful output value.
        int HowFar = CheckLocationDistance(akOldLoc, akNewLoc) + 2 
        ;GetRealDistance(akNewLoc, akOldLoc) ; Don't need this at all really - I will just use the sender if it is available
        If !JumpWasIntrasystem
            bool Damage = HandleEmergencyJump()
            ;DBG("HandleLocChange System change detected with Distance:" + HowFar)
            If !Damage
                HandleDrawingFuel(HowFar)
            Else
                ForceDrainFuel(true)
                DialogueShipServices.UpdateFuelGlobals() ; I shouldn't use forcedrain here technically so we have to notify other systems
            EndIf
        Else
            ;DBG("HandleLocChange detected inter-system travel. Standing by.")
        EndIf
        CheckRestrictions() ; This also fires on inter-system travel, takeoff, etc when placed here. Should still be on pilot's seat however.
        CheckAltActions()
        If DEBUGON
            GetGravRange() ; Just to see what's going on
        EndIf
        FuelToDrawSender = 0 ; If I reset this after calcs it should proc when refilled
        SpoolingUp = False
        MyOldShipSteal = False
    EndIf
EndFunction

; Event received when a ship initiates and completes docking with a parent
; This will be used for refueling
Function HandleDocking(bool abComplete, SpaceshipReference akDocking, SpaceshipReference akParent)
    If RF
        GetShip()
        If myShip == akDocking
            SiphonedShip = akParent
            Faction SiphonCrimeFaction = akParent.GetCrimeFaction()
            DBG("Ship crime faction is " + SiphonCrimeFaction)
            float myfuel = akDocking.GetValue(SpaceshipGravJumpFuel)
            float theirfuel = akParent.GetValue(SpaceshipGravJumpFuel)
            float fuel2rand = Math.Clamp( ( ( theirfuel * 0.75 ) * GetVariance(0.4) ), 25, 1000 )
            DBG("Detected dock with akDocking: " + akDocking + " having " + myfuel + "and akParent " + akParent + " having " + theirfuel + " randomized to " + fuel2rand )
            SiphonableFuel = fuel2rand
        EndIF
    EndIf
EndFunction

; Sanity check to clear even if no locchange occured
Function HandleUndocking(bool abComplete, SpaceshipReference akDocking, SpaceshipReference akParent)
    DBG("Running Siphon info clear on HandleUndocking")
    SiphonClear()
    DBG( "SiphonShip should be None:" + SiphonedShip + "with amount 0: " + SiphonableFuel )
EndFunction

; Event received when a ship begins or ends far travel - State { Departure = 0, Arrival = 1 }
Function HandleFarTravel(Location aDepartureLocation, Location aArrivalLocation, int aState)
    
EndFunction

; Event received when a ship grav jump event occurs - State { Initiated = 0, AnimStarted = 1, Completed = 2, Failed = 3 }
Function HandleGravJump(Location aDestination, int aState)
    ;DBG("HandleGravJump: fired with State " + aState)
    If aState
        If aState == ( 0 || 1 ) ; This will trigger our state to avoid clearing map set value!
            DBG("HandleGravJump: spoolup detected")
            SpoolingUp = True
        Else
            DBG("HandleGravJump: complete or failure detected")
            SpoolingUp = False
            FuelToDrawSender = 0 ; This would not have cleared on a successful jump from OnOpenGalaxy etc.
        EndIf
    EndIf
EndFunction

; Event that is triggered when fuel has been added to this spaceship
; Check if this triggers with our fuel consumable before hooking up the script. It didn't seem to.
Function HandleRefuel(int aFuelAdded)
    DBG("Refuel for " + aFuelAdded + "Detected")
    DialogueShipServices.UpdateFuelGlobals()
EndFunction

; Event received when a ship initiates or completes takeoff
Function HandleShipTakeOff(bool abComplete)
    GetShip()
EndFunction

Event Actor.OnSit(Actor akSender, ObjectReference akFurniture)
    If akSender == PlayerRef
        If akFurniture.HasRefType(Ship_PilotSeat_RefType)
            Utility.Wait(3) ; Not sure how long takeover scripts take to run  
            DBG("OnSit detected on pilot's seat")  
            SpaceshipReference ThisShip = PlayerRef.GetCurrentShipRef()
            If ThisShip != MyShip
                SpaceshipReference NewShip = ThisShip
                DBG("Player probably sat in a new ship which is " + NewShip + " vs old " + MyShip)
                SiphonedShip = MyShip
                float NewShipFuel = NewShip.GetValue(SpaceshipGravJumpFuel) + 5
                float OldShipFuel = MyShip.GetValue(SpaceshipGravJumpFuel) + 5
                DBG("Newship has " + NewShipFuel + " and old ship has " + OldShipFuel)
                SiphonableFuel = OldShipFuel
                DBG("We can siphon from our old ship as " + SiphonedShip + "with fuelvalue " + SiphonableFuel)
                Float RandomFillLevel = Utility.RandomFloat(0.35, 0.8)
                float NewShipRoll = Math.Clamp( ( ( NewShipFuel * RandomFillLevel ) * GetVariance(0.15) ), 0, 1000 )
                GetShip()
                If MyShip == NewShip
                    DBG("GetShip success! with MyShip " + MyShip + " and NewShip " + NewShip)
                    ForceDrainFuel(true)
                    Realfuel(false, NewShipRoll)
                    int Percent = GetFuelLevelInt()
                    _RF_State_CanActivateCargoPanel.SetValue(1)
                    MyOldShipSteal = True
                    Utility.Wait(0.75)
                    Debug.Notification("This ship has " + NewShipRoll as int + " He-3.")
                EndIF
            EndIf
        EndIf
    EndIf
EndEvent



;-----------------------------------------------------------
;----------------- EXTERNAL FUNCTION CALLS -----------------
;-----------------------------------------------------------

;This returns true if we have at least one
bool Function CheckHold(Form asFormToCheck)
    bool HasItem = false
    ;GetShip() - If we're running this we are in the ship - probably safe to skip
    If MyShip && asFormToCheck
        int AmountInHold = MyShip.GetItemCount(asFormToCheck)
        If  AmountInHold > 0
            HasItem = True
        EndIf
        DBG("CheckHold ran for " + asFormToCheck + " returning " + AmountInHold )
    EndIf
    Return HasItem
EndFunction

Function HandleReturn(Form asFormToReturn, bool abWaste = false)
    If !abWaste
        PlayerRef.AddItem(asFormToReturn, 1, true)
    Else
        MyShip.AddItem(asFormToReturn, 1, true)
    EndIF
    If !RF
        Debug.Notification("This cannot be used right now.")
    EndIF
EndFunction

Bool Function CheckInShip()
    Bool ShipOK = false
    ShipOK = _RF_COND_PlayerShipInterior.IsTrue(PlayerRef) 
    DBG("CheckInShip returned " + ShipOK + " with location " + PlayerRef.GetCurrentLocation() ) 
    Return ShipOK
EndFunction

; Returns float value of the fuel in our tanks.
; If called true, returns float value of empty space in tanks
float Function CheckFuelAmount(bool abAmountMissing = false)
    float Fuel = 1
    If !abAmountMissing
        Fuel =  MyShip.GetValue(SpaceshipGravJumpFuel)
        DBG("CheckFuelAmount: Current got " + Fuel)
    Else
        Fuel = MyShip.GetBaseValue(SpaceshipGravJumpFuel) - MyShip.GetValue(SpaceshipGravJumpFuel)
        DBG("CheckFuelAmount: Missing got " + Fuel)
    EndIF
    Return Fuel
EndFunction

Function HandleSettings()
    int Mode = _RF_Sys_ModEnabled.GetValue() as int
    int Travel = _RF_Sys_AllowTravel.GetValue() as int
    int Verbose = _RF_Sys_Verbose.GetValue() as int ; This gets checked each call - whatever
    If Mode != 1
        RF = False
        MyShip.RestoreValue(SpaceshipGravJumpFuel, 9999)
    EndIF
    If Travel == 1
        DisableFarTravel(false)
        DisableGravJump(false)
        DisableFastTravel(false)
        Debug.Notification("Travel restrictions removed.")
        PerkRefresh(false)
    EndIf
EndFunction

bool Function ModRunning()
    Return RF
EndFunction

Faction Function CheckShipCrime()
    Faction Controls
    GetShip()
    Location ShipIsIn = MyShip.GetCurrentLocation()
    If UCStarstations.Find(ShipIsIn) > 0
        Controls = CrimeFactionUC
    EndIF
    DBG("Ship is in " + ShipISIn + " for faction check " + Controls)
    Return Controls
EndFunction

Function SiphonFuel(bool abWithDevice = true)
    If RF
        bool Success = false
        DBG("SiphonFuel called externally. Found stored value " + SiphonableFuel + " and ship " + SiphonedShip)
        If SiphonableFuel > 0 && SiphonedShip
            Utility.Wait(0.75)
            int GoAhead = 0
            If !abWithDevice
                If !MyOldShipSteal
                    Debug.Notification("Docked vessel detected. Siphoning is available.")
                    Utility.Wait(3)
                    GoAhead = _RF_SiphonFuelWarning.Show()
                Else
                    Debug.Notification("Siphoning from former vessel...")
                    GoAhead = 1
                EndIF
            Else
                GoAhead = 1
                Debug.Notification("Docked vessel detected. Launching Siphon malware.")
            EndIf
            If GoAhead == 1
                DBG("SiphonFuel - Starting siphon procedure")
                Faction SiphonCrimeFaction = SiphonedShip.GetCrimeFaction()
                Utility.Wait(7)
                int Withdrawn = Realfuel(false, SiphonableFuel) ; This smart-clamps so we don't have to worry about overflow.
                Debug.Notification("Fuel siphoning successful: " + SiphonableFuel as int + " He-3 added to ship's tanks.")
                Success = True
                SiphonedShip.DamageValue( SpaceshipGravJumpFuel, ( Withdrawn - 5 ) )
                If !abWithDevice
                    TriggerBounty(SiphonCrimeFaction)
                Else
                    If MyOldShipSteal
                        HandleReturn(_RF_FuelSiphon)
                        Debug.Notification("I didn't need this to draw from my old vessel.")
                    EndIF
                EndIF
                SiphonClear()
                Utility.Wait(3)
                SmartNotify()
            Else
                Utility.Wait(2)
                Debug.Notification("Siphoning procedure cancelled.")
                SiphonClear()
            EndIf
        EndIf    
        If !Success
            If abWithDevice
                HandleReturn(_RF_FuelSiphon)
                Debug.Notification("Siphoning currently unavailable.")
            EndIf
        EndIF
    Else
        If abWithDevice
            HandleReturn(_RF_FuelSiphon)
        EndIf
    EndIF
EndFunction

Function HandleActivateCargoPanel()
    If !AutoRefuelRunning
    SoundWhenTriggeringSmartFuel.Play(PlayerRef)
    AutoRefuelRunning = True
        If GetFuelLevelInt() < BreakHigh || MyOldShipSteal
            If SiphonedShip
                SiphonFuel(false)
            Else
                bool HadFuelCanister = RefuelFromHold()
                If !HadFuelCanister
                    bool HadRawFuel = RefuelFromHoldManual()
                    If !HadRawFuel
                        Utility.Wait(3)
                        Debug.Notification("No available refuel methods found.")
                    EndIf
                EndIf
            EndIf
        EndIf
        CheckAltActions()
        Utility.Wait(12)
        AutoRefuelRunning = False
    Else
        Debug.Notification("Refuel checks already in progress...")
    EndIF
EndFunction

Function HandleActivateFuelContainer(ObjectReference akTargetRef)
    int He3InContainer = akTargetRef.GetItemCount(InorgCommonHelium3)
    DBG("Container has " + He3InContainer + "Helium") ; We will use this at 100% efficiency
    If He3InContainer > (MyShip.GetBaseValue(SpaceshipGravJumpFuel) * 0.10)
        Debug.Notification("He-3 found. Beginning transfer.")
        int FuelGotten = Realfuel(false, He3InContainer)
        akTargetRef.RemoveItem(InorgCommonHelium3, FuelGotten)
        Utility.Wait(4)
        Debug.Notification("Refueling procedure complete.")
        Utility.Wait(2)
        SmartNotify(GetFuelLevelInt())
        ;DialogueShipServices.UpdateFuelGlobals() ; RealFuel handles this - Thanks realfuel!
    Else
        Utility.Wait(2)
        Debug.Notification("Insufficient He-3 for Refueling.")
    EndIf
EndFunction